<?php

namespace App\Http\Controllers;

use App\Referral;
use App\User;
use App\Hospital;
use App\AssignedUser;
use App\Bed;
use App\Patient;
use App\Room;
use App\Station;
use App\Ambulance;
use App\Doctor;
use App\DepartmentBed;
use App\DoctorSchedule;
use App\ReferralLocation;
use App\Drrmo;
use App\Notification;
use Carbon\Carbon;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;

class ReferralController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    public function search(Request $request)
    {
        $data = [
            'hospitals' => Hospital::with('station')->orderBy('created_at', 'desc')->get()
        ];

        $ref_number = $request->ref_number;
        $data['ref_number'] = $ref_number;

        $hospital_id = $request->hospital_id;
        $data['hospital_id'] = $hospital_id;


        $ref_status = $request->ref_status;
        $data['ref_status'] = $ref_status;


        // $ref_by = $request->ref_by;
        // $data['ref_status'] = $ref_by;

        $date_from = $request->date_from;
        $data['date_from'] = $date_from;

        $date_to = $request->date_to;
        $data['date_to'] = $date_to;

        $data['is_search'] = false;

        $where = [];;

        if ($ref_number != '') {
            array_push($where, ['referrals.referral_no', 'like', '%' . $ref_number . '%']);

            $data['is_search'] = true;
        }
        if ($hospital_id != '') {
            array_push($where, ['referrals.ref_from_hospital', $hospital_id]);

            $data['is_search'] = true;
        }
        if ($ref_status != '') {
            array_push($where, ['referrals.ref_status', $ref_status]);

            $data['is_search'] = true;
        }
        if ($date_from != '' && $date_to != '' && $date_to > $date_from) {
            array_push($where, ['referrals.created_at', '>=', $date_from]);
            array_push($where, ['referrals.created_at', '<=', $date_to]);

            $data['is_search'] = true;
        }


        if (Auth::user()->role == 1) {
            if ($request->created == "on") {
                array_push($where, ['referrals.ref_from_hospital', Auth::user()->hospital_id]);
                $data['is_search'] = true;
            } else {
                array_push($where, ['referrals.ref_to_hospital', Auth::user()->hospital_id]);
            }
        } else {
            array_push($where, ['referrals.is_internal', false]);
        }

        if (Auth::user()->role == 4) {
            array_push($where, ['referrals.ref_from_drrmo', Auth::user()->drrmo_id]);
        }



        if ($data['is_search'] == false) {
            return redirect('/referrals');
        }

        $data['referrals'] = Referral::where($where)
            ->orderBy('created_at', 'desc')
            ->get();

        return view('referrals.index', $data);
    }


    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $data = [
            'hospitals' => Hospital::with('station')->orderBy('created_at', 'desc')->get()
        ];

        switch (Auth::user()->role) {
            case '1':
                $data['referrals'] =
                    Referral::where([['referrals.ref_to_hospital', Auth::user()->hospital_id]])
                    ->whereYear('created_at', now()->year)
                    ->orderBy('created_at', 'desc')
                    ->paginate(10);
                $data['notifs'] = Notification::where([['receiver_key', Auth::user()->hospital_id], ['is_seen', false]])->get();
                break;

            case '4':
                $data['referrals'] =
                    Referral::where([['referrals.ref_from_drrmo', Auth::user()->drrmo_id]])
                    ->whereYear('created_at', now()->year)
                    ->orderBy('created_at', 'desc')
                    ->paginate(10);
                $data['notifs'] = Notification::where([['receiver_key', Auth::user()->drrmo_id], ['is_seen', false]])->get();
                break;

            default:
                $data['referrals'] = Referral::where('is_internal', false)->whereYear('created_at', now()->year)->orderBy('created_at', 'desc')->paginate(10);
                // $data['notifs'] = Notification::where([ ['receiver_role', 2], ['is_seen', false] ])->get();

                if (Auth::user()->role == 2) {
                    $data['notifs'] = Notification::where([['receiver_role', 2], ['is_seen', false]])->get();
                }

                break;
        }


        return view('referrals.index', $data);
    }

    public function incoming()
    {
        $role = Auth::user()->role;

        if ($role == 2 || $role == 3) {
            $data = [
                'hospitals' => Hospital::with('station')->orderBy('created_at', 'desc')->get(),
                'referrals'  => Referral::select('referrals.*', 'hospitals.hospital')
                    ->join('hospitals', 'hospitals.id', '=', 'referrals.ref_from_hospital')
                    ->where(
                        [
                            ['referrals.ref_status', 2]
                        ]
                    )
                    ->orderBy('referrals.created_at', 'desc')
                    ->get()
            ];

            return view('referrals.index', $data);
        } else {
            $hospital_id = Auth::user()->hospital_id;
            $data = [
                'hospitals' => Hospital::with('station')->orderBy('created_at', 'desc')->get(),
                'referrals'  => Referral::select('referrals.*', 'hospitals.hospital')
                    ->join('hospitals', 'hospitals.id', '=', 'referrals.ref_from_hospital')
                    ->where(
                        [
                            ['referrals.ref_to_hospital', $hospital_id],
                            ['referrals.ref_status', 2]
                        ]
                    )
                    ->orderBy('referrals.created_at', 'desc')
                    ->get()
            ];
            return view('referrals.index', $data);
        }
    }

    public function on_process()
    {
        $role = Auth::user()->role;

        if ($role == 2 || $role == 3) {
            $data = [
                'hospitals' => Hospital::with('station')->orderBy('created_at', 'desc')->get(),
                'referrals'  => Referral::select('referrals.*', 'hospitals.hospital')
                    ->join('hospitals', 'hospitals.id', '=', 'referrals.ref_from_hospital')
                    ->where(
                        [
                            ['referrals.ref_to_hospital', null],
                            ['referrals.ref_status', 1]
                        ]
                    )
                    ->orderBy('referrals.created_at', 'desc')
                    ->get()
            ];
            return view('referrals.index', $data);
        } else {
            $hospital_id = Auth::user()->hospital_id;
            $data = [
                'hospitals' => Hospital::with('station')->orderBy('created_at', 'desc')->get(),
                'referrals'  => Referral::select('referrals.*', 'hospitals.hospital')
                    ->join('hospitals', 'hospitals.id', '=', 'referrals.ref_from_hospital')
                    ->where(
                        [
                            ['referrals.ref_from_hospital', $hospital_id],
                            ['referrals.ref_status', 1]
                        ]
                    )
                    ->orderBy('referrals.created_at', 'desc')
                    ->get()
            ];
            return view('referrals.index', $data);
        }
    }

    public function done()
    {
        $role = Auth::user()->role;

        if ($role == 2 || $role == 3) {
            $data = [
                'hospitals' => Hospital::with('station')->orderBy('created_at', 'desc')->get(),
                'referrals'  => Referral::select('referrals.*', 'hospitals.hospital')
                    ->join('hospitals', 'hospitals.id', '=', 'referrals.ref_from_hospital')
                    ->where(
                        [
                            ['referrals.ref_status', 3]
                        ]
                    )
                    ->orderBy('referrals.created_at', 'desc')
                    ->get()
            ];
            return view('referrals.index', $data);
        } else {
            $hospital_id = Auth::user()->hospital_id;
            $data = [
                'hospitals' => Hospital::with('station')->orderBy('created_at', 'desc')->get(),
                'referrals'  => Referral::select('referrals.*', 'hospitals.hospital')
                    ->join('hospitals', 'hospitals.id', '=', 'referrals.ref_from_hospital')
                    ->where(
                        [
                            ['referrals.ref_from_hospital', $hospital_id],
                            ['referrals.ref_status', 3]
                        ]
                    )
                    ->orderBy('referrals.created_at', 'desc')
                    ->get()
            ];
            return view('referrals.index', $data);
        }
    }

    public function create_patient()
    {
        $patient = array(
            'referral_id'   => $referral->id,
            'status'        =>  2,
            'hospital_id'   => $bed->hospital_id,
            'station_id'    => $bed->station_id,
            'room_id'       => $bed->room_id,
            'bed_id'        => $bed->id,
            'bed_no'        => $bed->bed
        );
        $patient = Patient::create($patient);
    }

    public function patient_arrived(Referral $referral)
    {
        $referral->ref_status = 3;
        $referral->save();

        if ($referral->is_dept_bed) {
            DepartmentBed::where('ref_id', $referral->id)->update(
                [
                    'ref_id'    =>  $referral->id,
                    'ref_no'    =>  $referral->referral_no,
                    'status'    =>  3
                ]
            );
        } else {
            Patient::where('referral_id', $referral->id)->update(['status' => 3]);
            Bed::where('ref_no', $referral->referral_no)->update(['status' => 3]);
        }
        return redirect()->back()->with('referral.patient_arrived', true);
    }

    public function patient_released(Referral $referral, Request $request)
    {
        $patient = Patient::where('referral_id', $referral->id)->first();
        if ($patient != null) { // if patient record found
            Patient::where('referral_id', $referral->id)->update(['status' => 4]);
        }

        //set bed status to available
        if ($referral->is_dept_bed) { // if referral was assigned in dept beds
            DepartmentBed::where('ref_id', $referral->id)->update(
                [
                    'ref_id'    =>  null,
                    'ref_no'    =>  null,
                    'status'    =>  1
                ]
            );
        } else { // if referral was assigned in station beds
            Bed::where('ref_no', $referral->referral_no)->update([
                'patient_id'    =>  null,
                'ref_no'        =>  null,
                'status'        =>  1
            ]);
        }


        $referral->ref_status = 4;
        $referral->remarks = $request->remarks;
        $referral->save();

        return redirect()->back()->with('referral.released', true);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(Request $request)
    {
        // $hospital_id = AssignedUser::where('user_id', Auth::user()->id)->first();
        $role = Auth::user()->role;

        if ($role == 1) {
            $hospital_id = Auth::user()->hospital_id;
            $data = [
                'hospitals' => Hospital::with('station')->orderBy('created_at', 'desc')->get(),
                'hospital'  => Hospital::where('id', $hospital_id)->first()
            ];
        } elseif ($role == 4) {
            $drrmo_id = Auth::user()->drrmo_id;
            $data = [
                'hospitals' => Hospital::with('station')->orderBy('created_at', 'desc')->get(),
                'drrmo'  => Drrmo::where('id', $drrmo_id)->first()
            ];
        }

        if (isset($request->bed_id)) {
            $bed = Bed::where('id', $request->bed_id)->first();
            $data['bed'] = $bed;
            $data['room'] = Room::where('id', $bed->room_id)->first();
            $data['station'] = Station::where('id', $bed->station_id)->first();
        }

        if (isset($request->dept_bed_id)) {
            $data['dept_bed'] = DepartmentBed::where('id', $request->dept_bed_id)->first();
        }


        return view('referrals.create', $data);
    }

    public function direct_refer(Request $request)
    {
        $request->validate([
            'patient'      => 'required',
            'age'               => 'required',
            'sex'               => 'required',
            'birthdate'        => 'required',
            'address'           => 'required',
            'guardian'          => 'required',
            'complaint'         => 'required',
            'diagnosis'         => 'required',
            'reason'            => 'required',
            'ref_by'            => 'required',
            'contact_no'        => 'required',
            'referral_type'     =>  'required'
        ]);

        $ref_ctr = DB::table('referrals')->latest('ref_ctr')->first();
        $ref_ctr = $ref_ctr != null ? strval($ref_ctr->ref_ctr + 1) : '1';
        $ref_len = 6 - strlen($ref_ctr);
        $ref_zero = '';
        for ($i = 0; $i < $ref_len; $i++) {
            $ref_zero .= '0';
        }
        $ref_no = strtoupper(Auth::user()->lname) . '-' . $ref_zero . $ref_ctr;

        $data = request()->all();

        if ($request->vaccine_1st_dose == 'on') {
            $data['vaccine_1st_dose'] = true;
        }
        if ($request->vaccine_2nd_dose == 'on') {
            $data['vaccine_2nd_dose'] = true;
        }
        if ($request->rtpcr == '+') {
            $data['rtpcr'] = true;
        } elseif ($request->rtpcr == '-') {
            $data['rtpcr'] = false;
        } else {
            $data['rtpcr'] = null;
        }

        if ($request->rat == '+') {
            $data['rat'] = true;
        } elseif ($request->rat == '-') {
            $data['rat'] = false;
        } else {
            $data['rat'] = null;
        }

        $data['referral_no'] = $ref_no;
        $data['ref_ctr'] = intval($ref_ctr);
        $data['ref_from_date'] = now();
        $data['ref_from_time'] = now();
        $data['created_by'] = Auth::user()->id;
        $data['ref_status'] = 2;
        $data['is_internal'] = false;
        // dd($data);

        if ($request->rtpcr_attach !== null) {
            $fileName = time() . '-' . $ref_no . '.' . $request->rtpcr_attach->extension();
            $request->rtpcr_attach->move(public_path('rtpcr-attachments'), $fileName, 0777);
            $data['rtpcr_attach'] = $fileName;
        }

        if ($request->rat_attach !== null) {
            $fileName = time() . '-' . $ref_no . '.' . $request->rat_attach->extension();
            $request->rat_attach->move(public_path('rat-attachments'), $fileName, 0777);
            $data['rat_attach'] = $fileName;
        }

        $referral = Referral::create($data);

        // $referral = Referral::create(array_merge(request()->all(), ['referral_no' => $ref_no, 'ref_ctr' => intval($ref_ctr), 'ref_status' => 2, 'ref_to_date' => now(), 'ref_to_time' => now(), 'ref_from_date' => now(), 'ref_from_time' => now(), 'created_by' => Auth::user()->id]));
        AuditTrailController::store(
            'referrals',
            'create',
            'Create referrals, ' . $request->patient_name . ' REF # ' . $referral->referral_no
        );

        $hod_name = Auth::user()->fname . ' ' . Auth::user()->lname;

        // $type, $notif, $link, $rec_role, $rec_key, $ref_no
        NotificationController::store(
            'referral',
            ucwords($hod_name) . ' created a referral [' . $referral->referral_no . ']',
            '/referral/' . $referral->id,
            '1',
            $referral->ref_to_hospital,
            $referral->referral_no,
            null
        );

        if (isset($request->bed_id)) {
            $bed = Bed::where('id', $request->bed_id)->first();
            $patient = array(
                'referral_id'   => $referral->id,
                'status'        =>  2,
                'hospital_id'   => $bed->hospital_id,
                'station_id'    => $bed->station_id,
                'room_id'       => $bed->room_id,
                'bed_id'        => $bed->id,
                'bed_no'        => $bed->bed
            );
            $patient = Patient::create($patient);

            $bed->status = 2;
            $bed->patient_id = $patient->id;
            $bed->ref_no = $referral->referral_no;
            $bed->save();
            $referral->ref_to_hospital = $bed->hospital_id;
            $referral->is_dept_bed = false;
            $referral->save();
            return redirect('/referral/' . $referral->id)->with('referral.store', true);
        } else {
            $dept_bed = DepartmentBed::where('id', $request->dept_bed_id)->first();
            $dept_bed->status = 2;
            $dept_bed->ref_id = $referral->id;
            $dept_bed->ref_no = $referral->referral_no;
            $dept_bed->save();
            $referral->ref_to_hospital = $dept_bed->hospital_id;
            $referral->is_dept_bed = true;
            $referral->save();
            return redirect('/referral/' . $referral->id)->with('referral.store', true);
        }
    }


    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request->validate([
            'patient'      => 'required',
            'age'               => 'required',
            'sex'               => 'required',
            'birthdate'        => 'required',
            'address'           => 'required',
            'guardian'          => 'required',
            'complaint'         => 'required',
            'diagnosis'         => 'required',
            'reason'            => 'required',
            'ref_by'            => 'required',
            'contact_no'        => 'required',
            'referral_type'     =>  'required'
        ]);

        if (isset($request->ref_from_hospital)) {
            $hospital_code = Hospital::where('id', '=', $request->ref_from_hospital)->first();
            $ref_ctr = DB::table('referrals')->latest('ref_ctr')->first();
            $ref_ctr = $ref_ctr != null ? strval($ref_ctr->ref_ctr + 1) : '1';
            $ref_len = 6 - strlen($ref_ctr);
            $ref_zero = '';
            for ($i = 0; $i < $ref_len; $i++) {
                $ref_zero .= '0';
            }
            $ref_no = $hospital_code->code . '-' . $ref_zero . $ref_ctr;
        } elseif (isset($request->ref_from_drrmo)) {
            $drrmo_code = Drrmo::where('id', '=', $request->ref_from_drrmo)->first();
            $ref_ctr = DB::table('referrals')->latest('ref_ctr')->first();
            $ref_ctr = $ref_ctr != null ? strval($ref_ctr->ref_ctr + 1) : '1';
            $ref_len = 6 - strlen($ref_ctr);
            $ref_zero = '';
            for ($i = 0; $i < $ref_len; $i++) {
                $ref_zero .= '0';
            }
            $ref_no = $drrmo_code->code . '-' . $ref_zero . $ref_ctr;
        }

        $data = request()->all();

        if ($request->vaccine_1st_dose == 'on') {
            $data['vaccine_1st_dose'] = true;
        }
        if ($request->vaccine_2nd_dose == 'on') {
            $data['vaccine_2nd_dose'] = true;
        }
        if ($request->rtpcr == '+') {
            $data['rtpcr'] = true;
        } elseif ($request->rtpcr == '-') {
            $data['rtpcr'] = false;
        } else {
            $data['rtpcr'] = null;
        }

        if ($request->rat == '+') {
            $data['rat'] = true;
        } elseif ($request->rat == '-') {
            $data['rat'] = false;
        } else {
            $data['rat'] = null;
        }

        $data['referral_no'] = $ref_no;
        $data['ref_ctr'] = intval($ref_ctr);
        $data['ref_from_date'] = now();
        $data['ref_from_time'] = now();
        $data['created_by'] = Auth::user()->id;
        // dd($data);

        if ($request->rtpcr_attach !== null) {
            $fileName = time() . '-' . $ref_no . '.' . $request->rtpcr_attach->extension();
            $request->rtpcr_attach->move(public_path('rtpcr-attachments'), $fileName, 0777);
            $data['rtpcr_attach'] = $fileName;
        }

        if ($request->rat_attach !== null) {
            $fileName = time() . '-' . $ref_no . '.' . $request->rat_attach->extension();
            $request->rat_attach->move(public_path('rat-attachments'), $fileName, 0777);
            $data['rat_attach'] = $fileName;
        }

        $referral = Referral::create($data);

        // $id = DB::table('referrals')->latest('id')->first();
        if (isset($request->ref_from_hospital)) {
            // $type, $notif, $link, $rec_role, $rec_key, $ref_no
            NotificationController::store(
                'referral',
                ucwords($hospital_code->hospital) . ' created a referral [' . $referral->referral_no . ']',
                '/referral/' . $referral->id,
                '2',
                '0',
                $referral->referral_no,
                null
            );
        } elseif (isset($request->ref_from_drrmo)) {
            // $type, $notif, $link, $rec_role, $rec_key, $ref_no
            NotificationController::store(
                'referral',
                ucwords($drrmo_code->drrmo) . ' created a referral [' . $referral->referral_no . ']',
                '/referral/' . $referral->id,
                '2',
                '0',
                $referral->referral_no,
                null
            );
        }
        AuditTrailController::store(
            'referrals',
            'create',
            'Create referrals, ' . $request->patient_name . ' REF # ' . $referral->referral_no
        );


        return redirect('/referral/' . $referral->id)->with('referral.store', true);
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Referral  $referral
     * @return \Illuminate\Http\Response
     */
    public function show(Referral $referral)
    {
        // check if the hospital viewing is correct else it will redirect to referrals page
        if (Auth::user()->role == 1 || Auth::user()->role == 4) {
            $auth = Auth::user();
            $not_authorized = true;
            if ($auth->role == 1) {
                if ($auth->hospital_id == $referral->ref_from_hospital) $not_authorized = false;
                if ($auth->hospital_id == $referral->ref_to_hospital) $not_authorized = false;
            } else {
                if ($auth->drrmo_id == $referral->ref_from_drrmo) $not_authorized = false;
            }

            if ($not_authorized) return redirect('/referrals');
        }

        $data = [
            'hospitals' => Hospital::with('station')->orderBy('created_at', 'desc')->get(),
            'drrmos'    => Drrmo::get(),
            'referral'  => $referral
        ];

        if ($referral->is_dept_bed) {
            $data['dept_bed'] = DepartmentBed::where('ref_id', $referral->id)->first();
        } else {
            $patient = Patient::where('referral_id', $referral->id)->first();
            if ($patient != null) {
                $data['patient']    = $patient;
                $data['bed']        = Bed::where('id', $patient->bed_id)->first();
                $data['room']       = Room::where('id', $patient->room_id)->first();
                $data['station']    = Station::where('id', $patient->station_id)->first();
            }
            $bed = Bed::where('hospital_id', $referral->ref_to_hospital)->first();
        }

        if (isset($_GET['notif']) && Auth::user()->role != 3) {
            $notif_id = $_GET['notif'];
            $notif = Notification::where('id', $notif_id)->update(['is_seen' => true]);
        }

        $role = Auth::user()->role;
        if ($role == 1) {
            $data['notifs'] = Notification::where([['receiver_key', Auth::user()->hospital_id], ['is_seen', false]])->get();
        } elseif ($role == 2) {
            $data['notifs'] = Notification::where([['receiver_role', 2], ['is_seen', false]])->get();
        } elseif ($role == 4) {
            $data['notifs'] = Notification::where([['receiver_key', Auth::user()->drrmo_id], ['is_seen', false]])->get();
        }

        if (Auth::user()->role == 2) {
            $referral->is_seen = true;
            $referral->save();
        }


        return view('referrals.show', $data);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Referral  $referral
     * @return \Illuminate\Http\Response
     */
    public function edit(Referral $referral)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Referral  $referral
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Referral $referral)
    {
        $today = Carbon::today();
        $data = [
            'hospital'  => Hospital::where('id', $request->ref_to_hospital)->first(),
            'stations'  => Station::where('hospital_id', $request->ref_to_hospital)->get(),
            'rooms'     => Room::where('hospital_id', $request->ref_to_hospital)->get(),
            'beds'      => Bed::where('beds.hospital_id', $request->ref_to_hospital)
                ->select('beds.id', 'beds.bed', 'beds.status', 'beds.patient_id', 'beds.hospital_id', 'beds.station_id', 'beds.room_id', 'stations.station', 'rooms.room')
                ->join('stations', 'stations.id', '=', 'beds.station_id')
                ->join('rooms', 'rooms.id', '=', 'beds.room_id')
                ->get(),
            'referral_no'  => $referral->referral_no,
            'referral_id'  => $referral->id,
            'ambulances'     => Ambulance::where('hospital_id', $request->ref_to_hospital)->get(),
            'doctors'       => Doctor::where('hospital_id', $request->ref_to_hospital)->get(),
            'or_beds'   => DepartmentBed::where([
                ['hospital_id', $request->ref_to_hospital],
                ['type', 'or']
            ])->get(),
            'er_beds'        => DepartmentBed::where([
                ['hospital_id', $request->ref_to_hospital],
                ['type', 'er']
            ])->get(),
            'dr_beds'        => DepartmentBed::where([
                ['hospital_id', $request->ref_to_hospital],
                ['type', 'dr']
            ])->get(),
            'er_doctors' => DoctorSchedule::select('doctor_schedules.*', 'doctors.doctor', 'doctors.specialty')
                ->join('doctors', 'doctors.id', '=', 'doctor_schedules.doctor_id')
                ->where([
                    ['doctors.hospital_id', $request->ref_to_hospital],
                    ['doctor_schedules.department', 'er']
                ])
                ->whereDate('doctor_schedules.day', '=', $today->toDateString())
                ->get(),
            'opd_beds'        => DepartmentBed::where([
                ['hospital_id', $request->ref_to_hospital],
                ['type', 'opd']
            ])->get(),
            'dr_doctors' => DoctorSchedule::select('doctor_schedules.*', 'doctors.doctor', 'doctors.specialty')
                ->join('doctors', 'doctors.id', '=', 'doctor_schedules.doctor_id')
                ->where([
                    ['doctors.hospital_id', $request->ref_to_hospital],
                    ['doctor_schedules.department', 'dr']
                ])
                ->whereDate('doctor_schedules.day', '=', $today->toDateString())
                ->get(),
            'or_doctors' => DoctorSchedule::select('doctor_schedules.*', 'doctors.doctor', 'doctors.specialty')
                ->join('doctors', 'doctors.id', '=', 'doctor_schedules.doctor_id')
                ->where([
                    ['doctors.hospital_id', $request->ref_to_hospital],
                    ['doctor_schedules.department', 'or']
                ])
                ->whereDate('doctor_schedules.day', '=', $today->toDateString())
                ->get(),
            'opd_doctors' => DoctorSchedule::select('doctor_schedules.*', 'doctors.doctor', 'doctors.specialty')
                ->join('doctors', 'doctors.id', '=', 'doctor_schedules.doctor_id')
                ->where([
                    ['doctors.hospital_id', $request->ref_to_hospital],
                    ['doctor_schedules.department', 'opd']
                ])
                ->whereDate('doctor_schedules.day', '=', $today->toDateString())
                ->get()
        ];


        return view('hospitals.show', $data);
    }

    public function transfer_patient(Request $request)
    {
        echo $request->ref_id;
    }

    public function finish(Request $request)
    { // function for assigning referral to Hospital
        $referral = Referral::where('referral_no', $request->ref_no)->first();
        $referral->ref_to_date = now();
        $referral->ref_to_time = now();
        $referral->ref_status = 2;

        if (isset($request->bed_id)) { //if HOD selected station beds
            $bed = Bed::where('id', $request->bed_id)->first();
            $bed->status = 2;
            $bed->ref_no = $referral->referral_no;
            $referral->ref_to_hospital = $bed->hospital_id;
            $referral->is_dept_bed = false;
            $referral->save();

            // when HOD selected station beds, it will create a patient data
            $patient = array(
                'referral_id'   => $referral->id,
                'status'        =>  2,
                'hospital_id'   => $bed->hospital_id,
                'station_id'    => $bed->station_id,
                'room_id'       => $bed->room_id,
                'bed_id'        => $bed->id,
                'bed_no'        => $bed->bed
            );
            $patient = Patient::create($patient);
            $bed->patient_id = $patient->id;
            $bed->save();

            $hospital = Hospital::find($bed->hospital_id);
        } else { //if HOD selected emergency department
            $dept_bed = DepartmentBed::where('id', $request->dept_bed_id)->first();
            $dept_bed->status = 2;
            $dept_bed->ref_no = $referral->referral_no;
            $dept_bed->ref_id = $referral->id;
            $dept_bed->save();
            $referral->ref_to_hospital = $dept_bed->hospital_id;
            $referral->is_dept_bed = true;
            $referral->save();

            $hospital = Hospital::find($dept_bed->hospital_id);
        }

        // notify the receiving hospital
        // $type, $notif, $link, $rec_role, $rec_key, $ref_no
        NotificationController::store(
            'referral',
            'Incoming referral [' . $referral->referral_no . ']',
            '/referral/' . $referral->id,
            '1',
            $referral->ref_to_hospital,
            $referral->referral_no,
            null
        );

        // notify the referring hospital
        if ($referral->ref_from_hospital != null) {
            NotificationController::store(
                'referred',
                'Referral [' . $referral->referral_no . '] has been referred to ' . $hospital->hospital,
                '/referral/' . $referral->id,
                '1',
                $referral->ref_from_hospital,
                $referral->referral_no,
                null
            );
        }
        //notify the referring drrmo
        if ($referral->ref_from_drrmo != null) {
            NotificationController::store(
                'referred',
                'Referral [' . $referral->referral_no . '] has been referred to ' . $hospital->hospital,
                '/referral/' . $referral->id,
                '1',
                $referral->ref_from_drrmo,
                $referral->referral_no,
                null
            );
        }

        //add to audit trail
        AuditTrailController::store(
            'referrals',
            'assign',
            'Assign referral REF # ' . $request->ref_no . ' to ' . ucwords($hospital->hospital)
        );

        return redirect('/referral/' . $referral->id)->with('referral.update', true);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Referral  $referral
     * @return \Illuminate\Http\Response
     */
    public function destroy(Referral $referral)
    {
        //
    }

    public function changeHospital(Referral $referral)
    {
        // retrieve patient record
        $patients = Patient::where('referral_id', $referral->id)->get();
        //set bed status to available
        if ($referral->is_dept_bed) { // if referral was assigned in dept beds
            DepartmentBed::where('ref_id', $referral->id)->update(
                [
                    'ref_id'    =>  null,
                    'ref_no'    =>  null,
                    'status'    =>  1
                ]
            );
        } else { // if referral was assigned in station beds
            Bed::where('ref_no', $referral->referral_no)->update([
                'patient_id'    =>  null,
                'ref_no'        =>  null,
                'status'        =>  1
            ]);
        }

        Patient::where('referral_id', $referral->id)->delete();

        $referral->ref_status = 1;
        $referral->ref_to_hospital = null;
        $referral->save();

        return redirect('/referral/' . $referral->id)->with('transfer', 'Please select the hospital to which you would like to transfer REF #' . $referral->referral_no);
    }
}
